// --- Lógica de Sesión ---
const userSessionElement = document.getElementById('user-session');
function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' })
        .then(r => r.json()).then(d => {
            if (d.loggedIn) userSessionElement.innerHTML = `Hola, <strong>${d.nombre}</strong> | <a href="index.html">Volver</a>`;
        }).catch(e => console.error(e));
}
checkLoginStatus();

document.addEventListener('DOMContentLoaded', () => {
    const canvas = document.getElementById('boxCanvas');
    const ctx = canvas.getContext('2d');

    // Inputs UI
    const boxModelSelect = document.getElementById('boxModel');
    const sideWidthInput = document.getElementById('sideWidth');
    const boxHeightInput = document.getElementById('boxHeight');
    const bgColor = document.getElementById('bgColor');
    const patternColor = document.getElementById('patternColor');
    const bgImageInput = document.getElementById('bgImageInput');
    const repeatBg = document.getElementById('repeatBg');
    const strokeColor = document.getElementById('strokeColor');
    const mainImageInput = document.getElementById('mainImageInput');
    const imgScale = document.getElementById('imgScale');
    const imgYOffset = document.getElementById('imgYOffset');
    const boxText = document.getElementById('boxText');
    const fontSelector = document.getElementById('fontSelector');
    const textColor = document.getElementById('textColor');
    const downloadBtn = document.getElementById('downloadBtn');
    const instructionText = document.getElementById('instructionText');
    const patternBtns = document.querySelectorAll('.pattern-btn');

    let bgImgObj = null;
    let mainImgObj = null;
    let activePattern = 'none';

    // --- CONFIGURACIÓN DE ALTA RESOLUCIÓN ---
    // 118 px por cm es aprox 300 DPI (Alta calidad de impresión)
    const CM_TO_PX = 118; 
    const MARGIN = 100; 

    function generatePatternImage(type, color) {
        const patCanvas = document.createElement('canvas');
        const pCtx = patCanvas.getContext('2d');
        const size = 200; // Patrón más grande para alta calidad
        patCanvas.width = size;
        patCanvas.height = size;

        pCtx.clearRect(0,0,size,size);
        pCtx.fillStyle = color;

        if (type === 'dots') {
            pCtx.beginPath(); pCtx.arc(50, 50, 20, 0, Math.PI*2); pCtx.arc(150, 150, 20, 0, Math.PI*2); pCtx.fill();
        } 
        else if (type === 'stripes') {
            pCtx.fillRect(0, 0, 40, 200); pCtx.fillRect(100, 0, 40, 200);
        }
        else if (type === 'checkers') {
            pCtx.fillRect(0, 0, 100, 100); pCtx.fillRect(100, 100, 100, 100);
        }
        else if (type === 'cow') {
            drawBlob(pCtx, 40, 40, 30); drawBlob(pCtx, 160, 60, 40); drawBlob(pCtx, 80, 160, 35);
        }
        else if (type === 'hearts') {
            drawHeart(pCtx, 50, 50, 40, color); drawHeart(pCtx, 150, 150, 40, color);
        }
        else if (type === 'stars') {
            drawStar(pCtx, 50, 50, 5, 30, 15, color); drawStar(pCtx, 150, 150, 5, 30, 15, color);
        }
        else if (type === 'flowers') {
            drawFlower(pCtx, 50, 50, 30, color); drawFlower(pCtx, 150, 150, 30, color);
        }

        const img = new Image();
        img.src = patCanvas.toDataURL();
        img.onload = () => {
            bgImgObj = img;
            repeatBg.checked = true;
            draw();
        };
    }

    // Funciones auxiliares de dibujo
    function drawBlob(ctx, x, y, r) {
        ctx.beginPath(); ctx.arc(x, y, r, 0, Math.PI * 2);
        ctx.ellipse(x, y, r, r*0.8, Math.random(), 0, Math.PI*2); ctx.fill();
    }
    function drawHeart(ctx, x, y, size, color) {
        ctx.save(); ctx.translate(x, y); ctx.scale(size/100, size/100);
        ctx.beginPath(); ctx.moveTo(0, 0);
        ctx.bezierCurveTo(-50, -50, -100, 0, 0, 100);
        ctx.bezierCurveTo(100, 0, 50, -50, 0, 0); ctx.fill(); ctx.restore();
    }
    function drawStar(ctx, cx, cy, spikes, outerRadius, innerRadius, color) {
        let rot = Math.PI/2*3; let x=cx; let y=cy; let step=Math.PI/spikes;
        ctx.beginPath(); ctx.moveTo(cx, cy-outerRadius);
        for(let i=0; i<spikes; i++){
            x=cx+Math.cos(rot)*outerRadius; y=cy+Math.sin(rot)*outerRadius; ctx.lineTo(x,y); rot+=step;
            x=cx+Math.cos(rot)*innerRadius; y=cy+Math.sin(rot)*innerRadius; ctx.lineTo(x,y); rot+=step;
        }
        ctx.lineTo(cx, cy-outerRadius); ctx.closePath(); ctx.fill();
    }
    function drawFlower(ctx, x, y, radius, color) {
        ctx.save(); ctx.translate(x, y);
        for (let i = 0; i < 5; i++) {
            ctx.rotate(Math.PI * 2 / 5); ctx.beginPath(); ctx.arc(0, radius/2, radius/2, 0, Math.PI*2); ctx.fill();
        }
        ctx.beginPath(); ctx.arc(0, 0, radius/3, 0, Math.PI*2);
        ctx.globalCompositeOperation = "destination-out"; ctx.fill();
        ctx.globalCompositeOperation = "source-over"; ctx.restore();
    }

    // --- CÁLCULOS Y DIBUJO ---
    function calculateDimensions(model) {
        const W = parseFloat(sideWidthInput.value) * CM_TO_PX;
        const H = parseFloat(boxHeightInput.value) * CM_TO_PX;
        let totalW = 0, totalH = 0, topPadding = 0, offsetX = 0;

        if (model === 'hexagonal') {
            totalW = (W * 3) + (CM_TO_PX * 1.2); totalH = H + (W * 1.4); topPadding = W * 0.8;
        } else if (model === 'milkbox') {
            totalW = (W * 4) + (CM_TO_PX * 1.2); totalH = H + (W * 1.3); topPadding = W * 0.7;
        } else if (model === 'pyramid') {
            totalW = (W * 4) + (CM_TO_PX * 1.2); totalH = H + W; topPadding = 0; 
        } else if (model === 'popcorn') {
            const topW = W * 1.4; const diff = (topW - W) / 2; 
            totalW = (topW * 4) + (CM_TO_PX * 1.2); totalH = H; offsetX = diff; 
        } else if (model === 'cube') {
            totalW = (W * 4) + (CM_TO_PX * 1.2); totalH = (W * 3); topPadding = W; 
        }
        return { w: totalW, h: totalH, topPadding: topPadding, offsetX: offsetX };
    }

    function definePath(model, x, y) {
        const W = parseFloat(sideWidthInput.value) * CM_TO_PX;
        const H = parseFloat(boxHeightInput.value) * CM_TO_PX;
        const flap = 1.2 * CM_TO_PX; 

        if (model === 'hexagonal') {
            const pH = W * 0.8; const bH = W * 0.6;
            ctx.moveTo(x, y);
            drawPetalPath(ctx, x, y, W, pH); drawPetalPath(ctx, x + W, y, W, pH); drawPetalPath(ctx, x + W*2, y, W, pH);
            ctx.lineTo(x + W*3 + flap, y + (CM_TO_PX*0.5)); ctx.lineTo(x + W*3 + flap, y + H - (CM_TO_PX*0.5)); ctx.lineTo(x + W*3, y + H);
            ctx.lineTo(x + W*3 - (W*0.15), y + H + bH); ctx.lineTo(x + W*2 + (W*0.15), y + H + bH); ctx.lineTo(x + W*2, y + H);
            ctx.lineTo(x + W*2 - (W*0.15), y + H + bH); ctx.lineTo(x + W + (W*0.15), y + H + bH); ctx.lineTo(x + W, y + H);
            ctx.lineTo(x + W - (W*0.15), y + H + bH); ctx.lineTo(x + (W*0.15), y + H + bH); ctx.lineTo(x, y + H);
            ctx.lineTo(x, y);
        } else if (model === 'pyramid') {
            const baseY = y + H;
            ctx.moveTo(x, baseY);
            ctx.lineTo(x + W*0.5, y); ctx.lineTo(x + W, baseY); ctx.lineTo(x + W*1.5, y); ctx.lineTo(x + W*2, baseY);
            ctx.lineTo(x + W*2.5, y); ctx.lineTo(x + W*3, baseY); ctx.lineTo(x + W*3.5, y); ctx.lineTo(x + W*4, baseY);
            ctx.lineTo(x + W*4 + flap, baseY - (CM_TO_PX*0.7)); ctx.lineTo(x + W*4 + (flap*0.5), baseY + (CM_TO_PX*0.5)); ctx.lineTo(x + W*4, baseY);
            ctx.lineTo(x + W*3, baseY); ctx.lineTo(x + W*2, baseY); 
            ctx.lineTo(x + W*2, baseY + W); ctx.lineTo(x + W, baseY + W); ctx.lineTo(x + W, baseY);
            ctx.lineTo(x, baseY); ctx.closePath();
        } else if (model === 'popcorn') {
            const topW = W * 1.4; const diff = (topW - W) / 2;
            ctx.moveTo(x, y + H); ctx.lineTo(x - diff, y);
            const panelTopW = topW;
            for(let i=0; i<4; i++) { let currX = (x - diff) + (panelTopW * i); ctx.lineTo(currX + panelTopW, y); }
            const topRightX = (x - diff) + (topW * 4); const bottomRightX = x + (W * 4);
            ctx.lineTo(topRightX + flap, y + (CM_TO_PX*0.5)); ctx.lineTo(bottomRightX + flap, y + H - (CM_TO_PX*0.5)); ctx.lineTo(bottomRightX, y + H);
            ctx.lineTo(x, y + H); ctx.closePath();
        } else if (model === 'milkbox') {
            const roofH = W * 0.7; const baseH = W * 0.6;
            ctx.moveTo(x, y);
            ctx.lineTo(x, y - roofH); ctx.lineTo(x + W*4, y - roofH); ctx.lineTo(x + W*4, y);
            ctx.lineTo(x + W*4 + flap, y + (CM_TO_PX*0.2)); ctx.lineTo(x + W*4 + flap, y + H - (CM_TO_PX*0.2)); ctx.lineTo(x + W*4, y + H);
            ctx.lineTo(x + W*4, y + H + baseH); ctx.lineTo(x + W*3, y + H + baseH); ctx.lineTo(x + W*3, y + H);
            ctx.lineTo(x + W*3, y + H + baseH); ctx.lineTo(x + W*2, y + H + baseH); ctx.lineTo(x + W*2, y + H);
            ctx.lineTo(x + W*2, y + H + baseH); ctx.lineTo(x + W, y + H + baseH); ctx.lineTo(x + W, y + H);
            ctx.lineTo(x + W, y + H + baseH); ctx.lineTo(x, y + H + baseH); ctx.lineTo(x, y + H);
            ctx.lineTo(x, y); ctx.closePath();
        } else if (model === 'cube') {
            ctx.moveTo(x, y); ctx.lineTo(x + W*2, y);
            ctx.lineTo(x + W*2, y - W + (CM_TO_PX*0.3)); ctx.lineTo(x + W*3, y - W + (CM_TO_PX*0.3)); ctx.lineTo(x + W*3, y);
            ctx.lineTo(x + W*4, y);
            ctx.lineTo(x + W*4 + flap, y + (CM_TO_PX*0.2)); ctx.lineTo(x + W*4 + flap, y + W - (CM_TO_PX*0.2)); ctx.lineTo(x + W*4, y + W);
            ctx.lineTo(x + W*3, y + W);
            ctx.lineTo(x + W*3, y + W*2 + (CM_TO_PX*0.3)); ctx.lineTo(x + W*2, y + W*2 + (CM_TO_PX*0.3)); ctx.lineTo(x + W*2, y + W);
            ctx.lineTo(x, y + W); ctx.closePath();
        }
    }

    function drawInternalLines(model, x, y) {
        const W = parseFloat(sideWidthInput.value) * CM_TO_PX;
        const H = parseFloat(boxHeightInput.value) * CM_TO_PX;
        if (model === 'hexagonal') {
            for (let i = 1; i <= 3; i++) { ctx.moveTo(x + W*i, y); ctx.lineTo(x + W*i, y + H); }
            ctx.moveTo(x, y); ctx.lineTo(x + W*3, y); ctx.moveTo(x, y+H); ctx.lineTo(x + W*3, y+H);
        } else if (model === 'pyramid') {
            const baseY = y + H;
            for (let i = 1; i <= 4; i++) { ctx.moveTo(x + W*i, baseY); ctx.lineTo(x + W*(i-0.5), y); }
            ctx.moveTo(x + W, baseY); ctx.lineTo(x + W*2, baseY); ctx.moveTo(x + W, baseY + W); ctx.lineTo(x + W*2, baseY + W);
        } else if (model === 'popcorn') {
            const topW = W * 1.4; const diff = (topW - W) / 2;
            for(let i=1; i<=3; i++) {
                const tx = (x - diff) + (topW * i); const bx = x + (W * i);
                ctx.moveTo(tx, y); ctx.lineTo(bx, y + H);
            }
            ctx.moveTo(x, y+H); ctx.lineTo(x+W*4, y+H);
        } else if (model === 'cube') {
            for(let i=1; i<=3; i++) ctx.moveTo(x + W*i, y), ctx.lineTo(x + W*i, y + W);
            ctx.moveTo(x, y); ctx.lineTo(x + W*4, y); ctx.moveTo(x, y+W); ctx.lineTo(x + W*4, y+W);
            ctx.moveTo(x + W*2, y - W); ctx.lineTo(x + W*3, y - W); ctx.moveTo(x + W*2, y + 2*W); ctx.lineTo(x + W*3, y + 2*W);
        } else if (model === 'milkbox') {
            const roofH = W * 0.7;
            for(let i=1; i<=3; i++) { ctx.moveTo(x + W*i, y - roofH); ctx.lineTo(x + W*i, y + H); }
            ctx.moveTo(x, y); ctx.lineTo(x + W*4, y); ctx.moveTo(x, y+H); ctx.lineTo(x + W*4, y+H);
            ctx.moveTo(x, y - roofH/2); ctx.lineTo(x + W*4, y - roofH/2);
        }
    }

    function drawDecorations(model, x, y) {
        const W = parseFloat(sideWidthInput.value) * CM_TO_PX;
        const H = parseFloat(boxHeightInput.value) * CM_TO_PX;
        let cx = 0, cy = 0;

        if (model === 'hexagonal') { cx = x + W*1.5; cy = y + H/2; }
        else if (model === 'milkbox') { cx = x + W*1.5; cy = y + H/2; } 
        else if (model === 'pyramid') { cx = x + W*1.5; cy = y + H - (H*0.4); } 
        else if (model === 'popcorn') { 
            const topW = W * 1.4; const diff = (topW - W) / 2;
            cx = x + W*1.5 + (diff*1.5); cy = y + H/2; 
        } else if (model === 'cube') { cx = x + W*1.5; cy = y + W/2; } 

        if (mainImgObj) {
            const scale = parseInt(imgScale.value) / 100;
            const offset = parseInt(imgYOffset.value) * (CM_TO_PX/45); // Ajustar offset a nueva escala
            const maxW = W * 0.9;
            let iW = mainImgObj.width; let iH = mainImgObj.height;
            if (iW > iH) { iW = maxW * scale; iH = (mainImgObj.height / mainImgObj.width) * iW; } 
            else { iH = (model==='pyramid' ? H*0.5 : H*0.7) * scale; iW = (mainImgObj.width / mainImgObj.height) * iH; }
            ctx.drawImage(mainImgObj, cx - iW/2, cy - iH/2 + offset, iW, iH);
        }

        if (boxText.value) {
            ctx.textAlign = "center";
            ctx.fillStyle = textColor.value;
            ctx.strokeStyle = "white";
            ctx.lineWidth = CM_TO_PX * 0.08;
            const fontSize = W * 0.18;
            ctx.font = `bold ${fontSize}px "${fontSelector.value}"`;
            const ty = cy + (H*0.35); 
            ctx.strokeText(boxText.value, cx, ty);
            ctx.fillText(boxText.value, cx, ty);
        }
    }

    function drawPetalPath(ctx, cx, cy, w, h) {
        ctx.bezierCurveTo(cx, cy - h*0.5, cx + w*0.5, cy - h, cx + w*0.5, cy - h);
        ctx.bezierCurveTo(cx + w*0.5, cy - h, cx + w, cy - h*0.5, cx + w, cy);
    }

    function draw() {
        const model = boxModelSelect.value;
        const dim = calculateDimensions(model);
        
        // Ajustar tamaño del canvas a la alta resolución
        canvas.width = dim.w + (MARGIN * 2);
        canvas.height = dim.h + (MARGIN * 2);

        // Instrucciones UI
        if (model === 'hexagonal') instructionText.innerText = "ℹ️ 2 Hojas. La base mide " + sideWidthInput.value + "cm.";
        else instructionText.innerText = "ℹ️ 1 Hoja. Medida Base: " + sideWidthInput.value + "cm.";

        ctx.clearRect(0, 0, canvas.width, canvas.height);
        const startX = MARGIN + dim.offsetX; 
        const startY = MARGIN + dim.topPadding;

        // MÁSCARA Y FONDO
        ctx.save();
        ctx.beginPath(); definePath(model, startX, startY); ctx.closePath();
        ctx.clip();
        ctx.fillStyle = bgColor.value;
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        if (bgImgObj) {
            if (repeatBg.checked) {
                const tempC = document.createElement('canvas');
                const tCtx = tempC.getContext('2d');
                let scalePat = activePattern === 'none' ? 0.4 : 1.0;
                tempC.width = bgImgObj.width * scalePat; 
                tempC.height = bgImgObj.height * scalePat;
                tCtx.drawImage(bgImgObj, 0, 0, tempC.width, tempC.height);
                const pattern = ctx.createPattern(tempC, 'repeat');
                ctx.fillStyle = pattern; ctx.fillRect(0, 0, canvas.width, canvas.height);
            } else {
                ctx.drawImage(bgImgObj, 0, 0, canvas.width, canvas.height);
            }
        }
        ctx.restore();

        // LÍNEAS DE CORTE Y DOBLADO
        ctx.lineWidth = 4; // Líneas más gruesas para alta res
        ctx.strokeStyle = strokeColor.value;
        ctx.beginPath(); definePath(model, startX, startY); ctx.stroke();

        ctx.beginPath(); ctx.setLineDash([20, 20]); 
        ctx.lineWidth = 2; ctx.strokeStyle = "#666"; 
        drawInternalLines(model, startX, startY); ctx.stroke(); ctx.setLineDash([]); 

        drawDecorations(model, startX, startY);
    }

    // --- LISTENERS ---
    const inputs = [boxModelSelect, sideWidthInput, boxHeightInput, bgColor, patternColor, repeatBg, strokeColor, imgScale, imgYOffset, boxText, fontSelector, textColor];
    inputs.forEach(el => { el.addEventListener('input', draw); el.addEventListener('change', draw); });

    patternBtns.forEach(btn => {
        btn.addEventListener('click', (e) => {
            patternBtns.forEach(b => b.classList.remove('active'));
            e.currentTarget.classList.add('active');
            const type = e.currentTarget.dataset.pattern;
            activePattern = type;
            if (type === 'none') { bgImgObj = null; draw(); } 
            else { generatePatternImage(type, patternColor.value); }
        });
    });

    patternColor.addEventListener('input', () => { if (activePattern !== 'none') generatePatternImage(activePattern, patternColor.value); });

    bgImageInput.addEventListener('change', (e) => {
        const f = e.target.files[0]; if(!f) return;
        const r = new FileReader();
        r.onload = (ev) => { 
            const img = new Image();
            img.onload = () => { bgImgObj = img; activePattern = 'none'; patternBtns.forEach(b => b.classList.remove('active')); draw(); };
            img.src = ev.target.result;
        };
        r.readAsDataURL(f);
    });

    mainImageInput.addEventListener('change', (e) => {
        const f = e.target.files[0]; if(!f) return;
        const r = new FileReader();
        r.onload = (ev) => { mainImgObj = new Image(); mainImgObj.onload = draw; mainImgObj.src = ev.target.result; };
        r.readAsDataURL(f);
    });

    // --- DESCARGA PDF PRECISA (CORREGIDA) ---
    downloadBtn.addEventListener('click', () => {
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF({ orientation: 'l', unit: 'mm', format: 'a4' });
        
        const imgData = canvas.toDataURL('image/jpeg', 0.9); // JPEG más ligero
        
        // Conversión Exacta: 1cm = 118px (definido arriba)
        // Factor: 10mm / 118px
        const pxToMm = 10 / CM_TO_PX;
        
        const pdfW = canvas.width * pxToMm;
        const pdfH = canvas.height * pxToMm;
        
        const pageW = 297; const pageH = 210;
        const x = (pageW - pdfW) / 2;
        const y = (pageH - pdfH) / 2;

        if (pdfW > pageW || pdfH > pageH) {
            alert("⚠️ Cuidado: El diseño es más grande que una hoja A4. Algunas partes se cortarán.");
        }

        doc.addImage(imgData, 'JPEG', x, y, pdfW, pdfH);
        
        doc.setFontSize(10);
        doc.text(`Molde: ${boxModelSelect.options[boxModelSelect.selectedIndex].text} - EspacioApp`, 10, 200);
        doc.save(`Molde_${boxModelSelect.value}.pdf`);
    });

    draw(); 
});