// --- Lógica de Sesión ---
const creditCounter = document.getElementById('creditCounterDisplay');
const userSessionElement = document.getElementById('user-session');

function updateCreditCounter(amount) {
    if (creditCounter) { creditCounter.textContent = `Créditos: ${amount}`; creditCounter.style.display = 'block'; }
}
function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' })
        .then(r => r.json()).then(d => {
            if (d.loggedIn) {
                userSessionElement.innerHTML = `Hola, <strong>${d.nombre}</strong> | <a href="index.html">Inicio</a> | <a href="logout.php">Salir</a>`;
                updateCreditCounter(d.creditos);
            } else { alert('Inicia sesión'); window.location.href = 'index.html'; }
        }).catch(e => console.error(e));
}
checkLoginStatus();

document.addEventListener('DOMContentLoaded', () => {
    const canvas = document.getElementById('bannerCanvas');
    const ctx = canvas.getContext('2d');

    // DOM Elements
    const bannerTextInput = document.getElementById('bannerText');
    const shapeSelector = document.getElementById('shapeSelector');
    const fontSelector = document.getElementById('fontSelector');
    const textColorPicker = document.getElementById('textColorPicker');
    const bgColorPicker = document.getElementById('bgColorPicker');
    const bgImageLoader = document.getElementById('bgImageLoader');
    const bannerHeightInput = document.getElementById('bannerHeightInput');
    const generatePdfBtn = document.getElementById('generatePdfBtn');
    const statusDiv = document.getElementById('status');
    const patternBtns = document.querySelectorAll('.pattern-btn');
    const patternColor = document.getElementById('patternColor');
    const borderWidthInput = document.getElementById('borderWidth');
    const borderColorInput = document.getElementById('borderColor');
    const showHolesCheckbox = document.getElementById('showHoles');
    const textSizeInput = document.getElementById('textSize');

    let bgImage = null;
    let activePattern = 'none';

    // Configuración Alta Calidad
    const BASE_WIDTH = 1000;
    const BASE_HEIGHT = 1400; 
    canvas.width = BASE_WIDTH;
    canvas.height = BASE_HEIGHT;

    // --- GENERADOR DE PATRONES ---
    function generatePatternImage(type, color) {
        const patCanvas = document.createElement('canvas');
        const pCtx = patCanvas.getContext('2d');
        const size = 200; 
        patCanvas.width = size; patCanvas.height = size;
        pCtx.clearRect(0,0,size,size);
        pCtx.fillStyle = color;

        if (type === 'dots') {
            pCtx.beginPath(); pCtx.arc(50, 50, 20, 0, Math.PI*2); pCtx.arc(150, 150, 20, 0, Math.PI*2); pCtx.fill();
        } else if (type === 'stripes') {
            pCtx.fillRect(0, 0, 40, 200); pCtx.fillRect(100, 0, 40, 200);
        } else if (type === 'checkers') {
            pCtx.fillRect(0, 0, 100, 100); pCtx.fillRect(100, 100, 100, 100);
        } else if (type === 'cow') {
            drawBlob(pCtx, 40, 40, 30); drawBlob(pCtx, 160, 60, 40); drawBlob(pCtx, 80, 160, 35);
        } else if (type === 'hearts') {
            drawHeart(pCtx, 50, 50, 40, color); drawHeart(pCtx, 150, 150, 40, color);
        } else if (type === 'stars') {
            drawStar(pCtx, 50, 50, 5, 30, 15, color); drawStar(pCtx, 150, 150, 5, 30, 15, color);
        } else if (type === 'flowers') {
            drawFlower(pCtx, 50, 50, 30, color); drawFlower(pCtx, 150, 150, 30, color);
        }

        const img = new Image();
        img.onload = () => { bgImage = img; drawBanner(getPreviewChar()); };
        img.src = patCanvas.toDataURL();
    }

    function drawBlob(ctx, x, y, r) { ctx.beginPath(); ctx.arc(x, y, r, 0, Math.PI * 2); ctx.ellipse(x, y, r, r*0.8, Math.random(), 0, Math.PI*2); ctx.fill(); }
    function drawHeart(ctx, x, y, size) { ctx.save(); ctx.translate(x, y); ctx.scale(size/100, size/100); ctx.beginPath(); ctx.moveTo(0,0); ctx.bezierCurveTo(-50,-50,-100,0,0,100); ctx.bezierCurveTo(100,0,50,-50,0,0); ctx.fill(); ctx.restore(); }
    function drawStar(ctx, cx, cy, spikes, outerRadius, innerRadius) { let rot=Math.PI/2*3; let x=cx; let y=cy; let step=Math.PI/spikes; ctx.beginPath(); ctx.moveTo(cx,cy-outerRadius); for(let i=0;i<spikes;i++){x=cx+Math.cos(rot)*outerRadius; y=cy+Math.sin(rot)*outerRadius; ctx.lineTo(x,y); rot+=step; x=cx+Math.cos(rot)*innerRadius; y=cy+Math.sin(rot)*innerRadius; ctx.lineTo(x,y); rot+=step;} ctx.lineTo(cx, cy-outerRadius); ctx.closePath(); ctx.fill(); }
    function drawFlower(ctx, x, y, radius) { ctx.save(); ctx.translate(x, y); for(let i=0;i<5;i++){ctx.rotate(Math.PI*2/5); ctx.beginPath(); ctx.arc(0, radius/2, radius/2, 0, Math.PI*2); ctx.fill();} ctx.beginPath(); ctx.arc(0,0,radius/3,0,Math.PI*2); ctx.globalCompositeOperation="destination-out"; ctx.fill(); ctx.globalCompositeOperation="source-over"; ctx.restore(); }

    function getPreviewChar() {
        const text = bannerTextInput.value.trim();
        return text.length > 0 ? text[0].toUpperCase() : "A";
    }

    // --- DEFINICIÓN DE FORMAS ---
    function defineBannerPath(ctx, w, h, shape) {
        ctx.beginPath();
        if (shape === 'triangle') {
            ctx.moveTo(0, 0); ctx.lineTo(w, 0); ctx.lineTo(w / 2, h); ctx.closePath();
        } else if (shape === 'swallowtail') {
            ctx.moveTo(0, 0); ctx.lineTo(w, 0); ctx.lineTo(w, h); ctx.lineTo(w / 2, h - (h * 0.25)); ctx.lineTo(0, h); ctx.closePath();
        } else if (shape === 'shield') {
            ctx.moveTo(0, 0); ctx.lineTo(w, 0); ctx.lineTo(w, h * 0.7); ctx.quadraticCurveTo(w / 2, h, 0, h * 0.7); ctx.closePath();
        } else {
            ctx.rect(0, 0, w, h);
        }
    }

    // --- DIBUJAR BANDERÍN (CORREGIDO: FONDO BLANCO) ---
    function drawBanner(letter) {
        const w = canvas.width;
        const h = canvas.height;
        const shape = shapeSelector.value;
        const font = fontSelector.value;
        const textColor = textColorPicker.value;
        const bgColor = bgColorPicker.value;
        const borderWidth = parseInt(borderWidthInput.value) * 2; 
        const borderColor = borderColorInput.value;

        // 1. CORRECCIÓN: PINTAR TODA LA HOJA DE BLANCO PRIMERO
        // Esto evita que las partes fuera del triángulo sean transparentes (negras en PDF)
        ctx.fillStyle = "#ffffff";
        ctx.fillRect(0, 0, w, h);

        // 2. Fondo y Patrón con Clipping (Recorte)
        ctx.save();
        defineBannerPath(ctx, w, h, shape);
        ctx.clip(); // Todo lo que se dibuje después, solo se verá dentro de la forma

        // Fondo del Banderín
        ctx.fillStyle = bgColor;
        ctx.fill();
        ctx.fillRect(0, 0, w, h); 

        // Patrón / Imagen
        if (bgImage) {
            if (activePattern !== 'none') {
                const pCanvas = document.createElement('canvas');
                pCanvas.width = bgImage.width; pCanvas.height = bgImage.height;
                pCanvas.getContext('2d').drawImage(bgImage, 0, 0);
                const pattern = ctx.createPattern(pCanvas, 'repeat');
                ctx.fillStyle = pattern;
                ctx.fillRect(0, 0, w, h);
            } else {
                const ratio = Math.max(w / bgImage.width, h / bgImage.height);
                const cx = (w - bgImage.width * ratio) / 2;
                const cy = (h - bgImage.height * ratio) / 2;
                ctx.drawImage(bgImage, 0, 0, bgImage.width, bgImage.height, cx, cy, bgImage.width * ratio, bgImage.height * ratio);
            }
        }
        ctx.restore(); // Dejamos de recortar

        // 3. Borde Interno
        if (borderWidth > 0) {
            ctx.save();
            defineBannerPath(ctx, w, h, shape);
            ctx.strokeStyle = borderColor;
            ctx.lineWidth = borderWidth;
            ctx.stroke();
            ctx.restore();
        }

        // 4. Guías de Perforación
        if (showHolesCheckbox.checked) {
            ctx.fillStyle = "white";
            ctx.strokeStyle = "#666";
            ctx.lineWidth = 2;
            const holeY = 50; 
            const holeMargin = 50; 
            
            ctx.beginPath(); ctx.arc(holeMargin, holeY, 15, 0, Math.PI*2); ctx.fill(); ctx.stroke();
            ctx.beginPath(); ctx.arc(w - holeMargin, holeY, 15, 0, Math.PI*2); ctx.fill(); ctx.stroke();
        }

        // 5. Texto
        if (letter) {
            ctx.fillStyle = textColor;
            const sizePct = parseInt(textSizeInput.value) / 100;
            const fontSize = h * 0.5 * sizePct;
            ctx.font = `bold ${fontSize}px "${font}"`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            
            ctx.shadowColor = "rgba(255, 255, 255, 0.8)";
            ctx.shadowBlur = 15;
            ctx.shadowOffsetX = 0; ctx.shadowOffsetY = 0;

            let yPos = h / 2;
            if (shape === 'triangle') yPos = h * 0.4;
            
            ctx.fillText(letter, w / 2, yPos);
            ctx.shadowBlur = 0; 
        }

        // 6. Línea de Corte (Para recortar con tijera)
        ctx.save();
        defineBannerPath(ctx, w, h, shape);
        ctx.strokeStyle = "#999"; // Gris para que se note sobre el blanco
        ctx.lineWidth = 2;
        ctx.setLineDash([15, 15]);
        ctx.stroke();
        ctx.restore();
    }

    // --- LISTENERS ---
    const inputs = [bannerTextInput, shapeSelector, fontSelector, textColorPicker, bgColorPicker, borderWidthInput, borderColorInput, showHolesCheckbox, textSizeInput];
    inputs.forEach(el => el.addEventListener('input', () => drawBanner(getPreviewChar())));

    patternBtns.forEach(btn => {
        btn.addEventListener('click', (e) => {
            patternBtns.forEach(b => b.classList.remove('active'));
            e.currentTarget.classList.add('active');
            const type = e.currentTarget.dataset.pattern;
            activePattern = type;
            if (type === 'none') { bgImage = null; drawBanner(getPreviewChar()); }
            else { generatePatternImage(type, patternColor.value); }
        });
    });

    patternColor.addEventListener('input', () => {
        if (activePattern !== 'none') generatePatternImage(activePattern, patternColor.value);
    });

    bgImageLoader.addEventListener('change', (e) => {
        const f = e.target.files[0]; if(!f) return;
        const r = new FileReader();
        r.onload = (ev) => {
            const img = new Image();
            img.onload = () => {
                bgImage = img;
                activePattern = 'none';
                patternBtns.forEach(b => b.classList.remove('active'));
                drawBanner(getPreviewChar());
            };
            img.src = ev.target.result;
        };
        r.readAsDataURL(f);
    });

    // --- GENERAR PDF ---
    generatePdfBtn.addEventListener('click', () => {
        const text = bannerTextInput.value.trim();
        if (!text) { alert("Escribe un texto."); return; }

        generatePdfBtn.disabled = true; generatePdfBtn.textContent = "Generando...";
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF({ orientation: 'p', unit: 'mm', format: 'a4' });
        const letters = text.split('').filter(c => c.trim() !== "");

        let hCM = parseFloat(bannerHeightInput.value);
        if (!hCM || hCM < 5) hCM = 20;
        const hMM = hCM * 10;
        const ratio = canvas.width / canvas.height;
        const wMM = hMM * ratio;

        const pageW = 210; 
        const marginX = (pageW - wMM) / 2;
        const marginY = 15;

        for (let i = 0; i < letters.length; i++) {
            if (i > 0) doc.addPage();
            drawBanner(letters[i].toUpperCase());
            const imgData = canvas.toDataURL('image/jpeg', 0.95);
            
            if(wMM > (pageW - 10)) {
                alert(`¡Ojo! El ancho (${(wMM/10).toFixed(1)}cm) es casi igual o mayor a la hoja. Podría cortarse.`);
            }

            doc.addImage(imgData, 'JPEG', marginX, marginY, wMM, hMM);
            doc.setFontSize(10);
            doc.setTextColor(150);
            doc.text(`Letra "${letters[i]}" - Alto: ${hCM}cm`, 10, 280);
            doc.text('espacioapp.site', 200, 280, {align: 'right'});
        }

        doc.save(`Banderines_${text}.pdf`);
        generatePdfBtn.disabled = false; generatePdfBtn.textContent = "📥 Generar PDF";
    });

    drawBanner("A");
});