document.addEventListener('DOMContentLoaded', () => {
    // Referencias a los elementos del HTML
    const galleryGrid = document.getElementById('gallery-grid');
    const loadingMessage = document.getElementById('gallery-loading-message');
    const searchInput = document.getElementById('searchInput');
    const searchBtn = document.getElementById('searchBtn');

    /**
     * Función para cargar proyectos desde el servidor.
     * @param {string} query - Texto opcional para filtrar (búsqueda).
     */
    function loadCommunityProjects(query = '') {
        // 1. Limpiar la rejilla actual y mostrar "Cargando..."
        galleryGrid.innerHTML = '';
        loadingMessage.style.display = 'block';
        loadingMessage.textContent = query ? 'Buscando...' : 'Cargando proyectos de la comunidad...';
        galleryGrid.appendChild(loadingMessage);

        // 2. Construir la URL (si hay texto, se añade ?q=texto)
        const url = query 
            ? `cargar_comunidad.php?q=${encodeURIComponent(query)}` 
            : 'cargar_comunidad.php';

        // 3. Pedir datos al servidor
        fetch(url, { cache: 'no-store' })
            .then(response => {
                if (!response.ok) throw new Error("Error en la respuesta del servidor");
                return response.json();
            })
            .then(data => {
                // Ocultar mensaje de carga
                loadingMessage.style.display = 'none'; 

                if (data.success && data.proyectos.length > 0) {
                    // 4. Si hay proyectos, crear las tarjetas (cards)
                    data.proyectos.forEach(proyecto => {
                        const proyectoCard = document.createElement('a');
                        proyectoCard.className = 'proyecto-card';
                        // Al hacer clic, envía al usuario al editor cargando este ID
                        proyectoCard.href = `poster.html?cargar_id=${proyecto.id}`;
                        
                        proyectoCard.innerHTML = `
                            <img src="${proyecto.url_miniatura}" alt="${proyecto.nombre_proyecto}" loading="lazy">
                            <h3>${proyecto.nombre_proyecto}</h3>
                            <p>${proyecto.ancho_cm}cm x ${proyecto.alto_cm}cm</p>
                            <p class="project-author">Por: ${proyecto.creador_nombre || 'Anónimo'}</p>
                        `;
                        galleryGrid.appendChild(proyectoCard);
                    });

                } else if (data.success && data.proyectos.length === 0) {
                    // Caso: No hay resultados
                    loadingMessage.style.display = 'block';
                    loadingMessage.textContent = query 
                        ? `No encontramos proyectos con el nombre "${query}".`
                        : 'Aún no hay proyectos públicos en la comunidad.';
                    galleryGrid.appendChild(loadingMessage);
                } else {
                    // Caso: Error lógico
                    loadingMessage.style.display = 'block';
                    loadingMessage.textContent = 'Error al cargar la lista de proyectos.';
                    galleryGrid.appendChild(loadingMessage);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                loadingMessage.style.display = 'block';
                loadingMessage.textContent = 'Error de conexión. Intenta recargar la página.';
                galleryGrid.appendChild(loadingMessage);
            });
    }

    // --- EVENTOS DE BÚSQUEDA ---
    
    // 1. Evento click en el botón "Buscar"
    if (searchBtn) {
        searchBtn.addEventListener('click', () => {
            const texto = searchInput.value.trim();
            loadCommunityProjects(texto);
        });
    }

    // 2. Evento tecla "Enter" en el campo de texto
    if (searchInput) {
        searchInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                const texto = searchInput.value.trim();
                loadCommunityProjects(texto);
            }
        });
    }

    // --- INICIO ---
    // Cargar todos los proyectos al entrar a la página
    loadCommunityProjects();
});